package com.agilex.healthcare.veteranappointment.datalayer.appointment;

import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.UUID;

import javax.annotation.Resource;
import javax.ws.rs.WebApplicationException;

import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.veteranappointment.utils.SpringBasedIntegrationRepo;
import com.agilex.healthcare.veteranappointment.dataservice.AppointmentRequestDataService;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequest;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequestDetailCode;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequestMessage;
import com.agilex.healthcare.veteranappointment.domain.VARAppointmentRequests;
import com.agilex.healthcare.veteranappointment.domain.VARDetailCode;
import com.agilex.healthcare.mobilehealthplatform.domain.DOBDate;
import com.agilex.healthcare.mobilehealthplatform.domain.Facility;
import com.agilex.healthcare.mobilehealthplatform.domain.Patient;
import com.agilex.healthcare.mobilehealthplatform.domain.filter.datefilter.DateFilterFactory;
import com.agilex.healthcare.veteranappointment.enumeration.AppointmentRequestProviderOption;
import com.agilex.healthcare.veteranappointment.enumeration.AppointmentRequestPurposeOfVisit;
import com.agilex.healthcare.veteranappointment.enumeration.AppointmentRequestStatus;
import com.agilex.healthcare.veteranappointment.enumeration.AppointmentRequestType;
import com.agilex.healthcare.veteranappointment.enumeration.AppointmentRequestVisitType;
import com.agilex.healthcare.veteranappointment.exception.InvalidAppointmentRequestException;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.NullChecker;

public class AppointmentRequestDataLayerRepoTest extends SpringBasedIntegrationRepo {

	@Resource
	AppointmentRequestDataLayerRepo datalayer;

	@Resource
    AppointmentRequestDataService appointmentRequestDataService;

	Date createdDate;
	String appointmentType;
	String visitType;
	String facilityName;
	String email;
	String phoneNumber;
	String status;
	String optionDate1;
	String optionTime1;
	String optionDate2;
	String optionTime2;
	String optionDate3;
	String optionTime3;
	String purposeOfVisit;
	String otherPurposeOfVisit;
	String providerId;
	String providerName;
	String providerPersonClass;
	String providerOption;
	boolean secondRequest;

	Set<String> bestTimetoCall;

	String testPatientId;

	VARAppointmentRequest appointmentRequest;

	private VARAppointmentRequest appointmentRequest1;

	private VARAppointmentRequest appointmentRequest2;

	private VARAppointmentRequest appointmentRequest3;

    private VARAppointmentRequest appointmentRequest4;

	public AppointmentRequestDataLayerRepoTest() {
		init();
	}

	protected void init() {
		appointmentType = AppointmentRequestType.MENTAL_HEALTH.getName();
		visitType = AppointmentRequestVisitType.OFFICE_VISIT.getName();
		facilityName = "Test EligibilityFacility Name";
		email = "test@agilex.com";
		phoneNumber = "555-555-5555";
		Calendar calendar = Calendar.getInstance();
		createdDate = calendar.getTime();
		calendar.add(Calendar.DATE, 3);
		optionDate1 = DateHelper.formatDate(calendar.getTime());
		optionTime1 = "AM";
		calendar.add(Calendar.DATE, 1);
		optionDate2 = DateHelper.formatDate(calendar.getTime());
		optionTime2 = "PM";
		calendar.add(Calendar.DATE, 2);
		optionDate3 = DateHelper.formatDate(calendar.getTime());
		optionTime3 = "AM";
		status = AppointmentRequestStatus.SUBMITTED.getName();
		purposeOfVisit = AppointmentRequestPurposeOfVisit.OTHER.getName();
		otherPurposeOfVisit = "Other purpose of visit custom";
		providerId = "PROV1";
		providerName = "John Smith";
		providerPersonClass = "Physician";
		providerOption = AppointmentRequestProviderOption.WILLING_TO_SEE_OTHER_PROVIDER.getName();
		secondRequest = true;

		bestTimetoCall = new HashSet<String>();
		bestTimetoCall.add("9 AM - 11 AM");
		bestTimetoCall.add("11 AM - 1 PM");
	}

	protected VARAppointmentRequest createBasicAppointmentRequest(String patientId, String firstName, String lastName, String ssn, DOBDate dateOfBirth) {
		String facilityAddress = "test address";
		String facilityCity = "test city";
		String facilityState = "test state";
		String facilityCode = "537";
		String parentSiteCode = "test parent site code";
		String facilityType = "test facility type";
		Facility facility = createFacility(facilityCode, facilityName, facilityType, facilityAddress, facilityCity, facilityState, parentSiteCode);

		VARAppointmentRequest appointmentRequest = new VARAppointmentRequest();

		appointmentRequest.setCreatedDate(createdDate);
		appointmentRequest.setAppointmentType(appointmentType);
		appointmentRequest.setVisitType(visitType);
		appointmentRequest.setFacility(facility);
		appointmentRequest.setEmail(email);
		appointmentRequest.setPhoneNumber(phoneNumber);
		appointmentRequest.setStatus(status);
		appointmentRequest.setOptionDate1(optionDate1);
		appointmentRequest.setOptionTime1(optionTime1);
		appointmentRequest.setOptionDate2(optionDate2);
		appointmentRequest.setOptionTime2(optionTime2);
		appointmentRequest.setOptionDate3(optionDate3);
		appointmentRequest.setOptionTime3(optionTime3);
		appointmentRequest.setBestTimetoCall(bestTimetoCall);
		appointmentRequest.setPurposeOfVisit(purposeOfVisit);
		appointmentRequest.setOtherPurposeOfVisit(otherPurposeOfVisit);
		appointmentRequest.setProviderId(providerId);
		appointmentRequest.setProviderName(providerName);
		appointmentRequest.setProviderPersonClass(providerPersonClass);

		appointmentRequest.setTextMessagingAllowed(false);
		appointmentRequest.setTextMessagingPhoneNumber("555-555-5555");

		providerOption = getRandomProviderOption();
		appointmentRequest.setProviderOption(providerOption);

		appointmentRequest.setLastUpdatedDate(new Date());

		appointmentRequest.setPatient(createPatient(patientId, firstName, lastName, ssn, dateOfBirth));
		appointmentRequest.setSecondRequest(secondRequest);

		return appointmentRequest;
	}

	protected Patient createPatient(String patientId, String firstName, String lastName, String ssn, DOBDate dateOfBirth) {
		Patient patient = new Patient();
		patient.setId(patientId);
		patient.setFirstName(firstName);
		patient.setLastName(lastName);
		patient.setSsn(ssn);
		patient.setDateOfBirth(dateOfBirth);

		return patient;
	}

	protected Facility createFacility(String facilityCode, String facilityName, String facilityType, String facilityAddress, String facilityCity, String facilityState, String parentSiteCode) {
		Facility facility = new Facility();
		facility.setAddress(facilityAddress);
		facility.setCity(facilityCity);
		facility.setFacilityCode(facilityCode);
		facility.setName(facilityName);
		facility.setParentSiteCode(parentSiteCode);
		facility.setState(facilityState);
		facility.setType(facilityType);

		return facility;
	}

	protected String getRandomProviderOption() {
		String result = "";
		Random random = new Random();
		if (random.nextInt(9) >= 5) {
			result = AppointmentRequestProviderOption.WILLING_TO_SEE_OTHER_PROVIDER.getName();
		}

		return result;
	}

	@Before
	public void setupData() throws Exception {
        super.setUp();

		testPatientId = UUID.randomUUID().toString();
		appointmentRequest1 = createBasicAppointmentRequest(testPatientId, "testUser", "testLastName", "123232345", new DOBDate(new Date()));
		String facilityAddress = "test address";
		String facilityCity = "test city";
		String facilityState = "test state";
		String facilityCode = "1000";
		String facilityName = "EligibilityFacility 1";
		String parentSiteCode = "test parent site code";
		String facilityType = "test facility type";
		Facility facility = createFacility(facilityCode, facilityName, facilityType, facilityAddress, facilityCity, facilityState, parentSiteCode);

		appointmentRequest1.setFacility(facility);
		appointmentRequest1.setStatus(AppointmentRequestStatus.CANCELLED.getName());
		addARDetailCode(appointmentRequest1, "DETCODE8");
		addARDetailCode(appointmentRequest1, "DETCODE7");

		facilityCode = "1001";
		facilityName = "EligibilityFacility 3";
		facility = createFacility(facilityCode, facilityName, facilityType, facilityAddress, facilityCity, facilityState, parentSiteCode);

		appointmentRequest2 = createBasicAppointmentRequest(testPatientId, "testUser", "testLastName", "123232345", new DOBDate(new Date()));
		appointmentRequest2.setFacility(facility);
		appointmentRequest2.setCreatedDate(DateHelper.parseDate("5/15/2012"));
		appointmentRequest2.setStatus(AppointmentRequestStatus.BOOKED.getName());
		addARDetailCode(appointmentRequest2, "DETCODE1");

		appointmentRequest3 = createBasicAppointmentRequest(testPatientId, "testUser", "testLastName", "123232345", new DOBDate(new Date()));
		appointmentRequest3.setFacility(facility);
		appointmentRequest3.setSecondRequestSubmitted(false);

		appointmentRequest1 = datalayer.saveAppointmentRequest(appointmentRequest1);
		appointmentRequest2 = datalayer.saveAppointmentRequest(appointmentRequest2);
		appointmentRequest3 = datalayer.saveAppointmentRequest(appointmentRequest3);


	}

	@After
	public void removeData() throws Exception {
		datalayer.deleteAppointmentRequest(appointmentRequest1);
		datalayer.deleteAppointmentRequest(appointmentRequest2);
		datalayer.deleteAppointmentRequest(appointmentRequest3);

        if(appointmentRequest4!=null){
            datalayer.deleteAppointmentRequest(appointmentRequest4);
        }

		super.tearDown();
	}

	private void addARDetailCode(VARAppointmentRequest appointmentRequest, String code) {

		VARDetailCode detailCode = new VARDetailCode();
		detailCode.setCode(code);

		List<VARAppointmentRequestDetailCode> arDetailCodes = appointmentRequest.getAppointmentRequestDetailCode();
		VARAppointmentRequestDetailCode arDetailCode = new VARAppointmentRequestDetailCode();
		arDetailCode.setCreatedDate(new Date());
		arDetailCode.setDetailCode(detailCode);
		arDetailCode.setUserId("john.smith");
		arDetailCodes.add(arDetailCode);
		appointmentRequest.setAppointmentRequestDetailCode(arDetailCodes);
	}

	@Test
	public void verifySaveAndFetchOfBasicAppointmentRequest() {
		String patientId = UUID.randomUUID().toString();

		saveAndCheckBasicAppointmentRequest(patientId);
		VARAppointmentRequest testAppointmentRequest = null;
		try {
			testAppointmentRequest = datalayer.fetchAppointmentRequest("fakepatientid, should not return appointment", appointmentRequest.getUniqueId());
		} catch (WebApplicationException e) {
		}
		assertNull(testAppointmentRequest);

		appointmentRequest = datalayer.fetchAppointmentRequest(patientId, appointmentRequest.getUniqueId());
		verifyAppointmentRequest(appointmentRequest);

		datalayer.deleteAppointmentRequest(appointmentRequest);
	}

	@Test
	public void appointmentRequestSecurityTest() {
		String patientId = UUID.randomUUID().toString();

		saveAndCheckBasicAppointmentRequest(patientId);
		appointmentRequest = datalayer.fetchAppointmentRequest(patientId, appointmentRequest.getUniqueId());
		appointmentRequest.setEmail("me@test.com");
		appointmentRequest.setPatientId("stolen");
		VARAppointmentRequest saved = datalayer.saveAppointmentRequest(appointmentRequest);
		assertNull(saved);

		appointmentRequest.setEmail(email);
		appointmentRequest.setPatientId(patientId);
		verifyAppointmentRequest(appointmentRequest);

		datalayer.deleteAppointmentRequest(appointmentRequest);
	}

	@Test
	public void appointmentRequestDeleteSecurityTest() {
		String patientId = UUID.randomUUID().toString();

		saveAndCheckBasicAppointmentRequest(patientId);
		appointmentRequest = datalayer.fetchAppointmentRequest(patientId, appointmentRequest.getUniqueId());
		appointmentRequest.setEmail("me@test.com");
		appointmentRequest.setPatientId("stolen");
		datalayer.deleteAppointmentRequest(appointmentRequest);

		appointmentRequest.setEmail(email);
		appointmentRequest.setPatientId(patientId);
		appointmentRequest = datalayer.fetchAppointmentRequest(patientId, appointmentRequest.getUniqueId());
		assertNotNull(appointmentRequest);
		verifyAppointmentRequest(appointmentRequest);

		datalayer.deleteAppointmentRequest(appointmentRequest);
	}

	@Test
	public void retrieveAppointmentRequestsByApplyingFilter() throws Exception {
		VARAppointmentRequests appointmentRequests = datalayer.fetchAppointmentRequests(null);
		assertTrue(appointmentRequests.size() > 0);
	}

	@Test
	public void retrieveAppointmentRequestsByApplyingFilterWithFacility1() throws Exception {
		AppointmentRequestFilter filter = new AppointmentRequestFilter();
		filter.setFacilityName("EligibilityFacility 1");
		VARAppointmentRequests appointmentRequests = datalayer.fetchAppointmentRequests(filter);
		assertEquals(1, appointmentRequests.size());
		assertEquals(2, appointmentRequests.get(0).getAppointmentRequestDetailCode().size());
	}

	@Test
	public void retrieveAppointmentRequestsByApplyingFilterWithFacility1AndCaseInsensitive() throws Exception {
		AppointmentRequestFilter filter = new AppointmentRequestFilter();
		filter.setFacilityName("facility 1");
		VARAppointmentRequests appointmentRequests = datalayer.fetchAppointmentRequests(filter);
		assertEquals(1, appointmentRequests.size());
	}

	@Test
	public void retrieveAppointmentRequestsByApplyingFilterWithFacility3() throws Exception {
		AppointmentRequestFilter filter = new AppointmentRequestFilter();
		filter.setFacilityName("EligibilityFacility 3");
		filter.setStartDate(DateHelper.parseDate("5/1/2012"));
		filter.setEndDate(DateHelper.parseDate("5/25/2012"));
		VARAppointmentRequests appointmentRequests = datalayer.fetchAppointmentRequests(filter);
		assertEquals(1, appointmentRequests.size());
		VARDetailCode detailCode = appointmentRequests.get(0).getAppointmentRequestDetailCode().iterator().next().getDetailCode();
		assertDetailCode(detailCode, "DETCODE1");
	}

	@Test
	public void validateSaveSecondAppointmentRequest() {
		VARAppointmentRequest secondAppointmentRequest = createBasicAppointmentRequest(testPatientId, "testUser", "testLastName", "123232345", new DOBDate(new Date()));
		secondAppointmentRequest.setParentRequest(appointmentRequest3);
		secondAppointmentRequest.setSecondRequest(true);
		appointmentRequest3.setSecondRequestSubmitted(true);

		VARAppointmentRequest savedSecondAppointmentRequest = datalayer.saveAppointmentRequest(secondAppointmentRequest);

		assertEquals(true, savedSecondAppointmentRequest.isSecondRequest());
		assertNull(savedSecondAppointmentRequest.getParentRequest());
		assertEquals(testPatientId, savedSecondAppointmentRequest.getPatientId());

		VARAppointmentRequest parentRequest = datalayer.fetchAppointmentRequest(testPatientId, appointmentRequest3.getUniqueId());
		assertEquals(true, parentRequest.isSecondRequestSubmitted());
	}

	private void assertDetailCode(VARDetailCode detailCode, String code) {
		assertTrue(NullChecker.isNotNullish(detailCode.getProviderMessage()));
		assertTrue(NullChecker.isNotNullish(detailCode.getVeteranMessage()));
		assertEquals(code, detailCode.getCode());
	}

	@Test
	public void verifySaveAndFetchAllAppointmentRequests() {
		String patientId = UUID.randomUUID().toString();

		saveAndCheckBasicAppointmentRequest(patientId);

		VARAppointmentRequests appointmentRequests = datalayer.fetchAppointmentRequestsByPatient(patientId, DateFilterFactory.createEmptyFilter());

		assertNotNull(appointmentRequests);
		assertTrue(appointmentRequests.size() > 0);
		assertTrue(appointmentRequests.contains(appointmentRequest));
	}

	@Test
	public void providerSeenAppointmentRequestFlagGetsUpdated(){
		assertEquals(false, appointmentRequest1.isProviderSeenAppointmentRequest());

		VARAppointmentRequest updatedAR = null;
		try {
			updatedAR = datalayer.markAppointmentRequestIsSeenByProvider("fake id, should fail", appointmentRequest1.getUniqueId());
		} catch (WebApplicationException e) {
		}
		assertNull(updatedAR);

		updatedAR = datalayer.markAppointmentRequestIsSeenByProvider(appointmentRequest1.getPatientId(), appointmentRequest1.getUniqueId());
		assertEquals(true, updatedAR.isProviderSeenAppointmentRequest());
	}

     @Test
     public void whenAddingMessageOnSeenAppointmentRequestAppointmentMustBeFirstOnSortList(){

         appointmentRequest4 = createBasicAppointmentRequest(testPatientId, "testUser", "testLastName", "123232345", new DOBDate(new Date()));
         appointmentRequest4.setCreatedDate(new Date());

         appointmentRequest4 = datalayer.saveAppointmentRequest(appointmentRequest4);
         checkNewAppointmentPresentAndLastOnList();
         checkUnreadMessageAndRequestReadBeforeSendsRequestToTopOfList();
         checkAfterReadingMessageRequestGoesBackToBottom();

     }

    @Test
    public void whenCreatingAppointmentWithMessageAppointmentMustBeLastOnSortList(){
        appointmentRequest4 = createBasicAppointmentRequest(testPatientId, "testUser", "testLastName", "123232345", new DOBDate(new Date()));
        appointmentRequest4.setCreatedDate(new Date());
        appointmentRequest4 = datalayer.saveAppointmentRequest(appointmentRequest4);

        checkUnreadMessageAndRequestNeverReadBeforeSendsRequestToBottomOfList();

    }


    @Test
    public void testSortingRulesForFetchAppointmentRequestByPatientId(){
        String patientId = UUID.randomUUID().toString();
        VARAppointmentRequest aptReq1 = null, aptReq2 = null;
        try{
            //first add booked request
            aptReq1 = createBasicAppointmentRequest(patientId, "testUser", "testLastName", "123232345", new DOBDate(new Date()));
            aptReq1.setStatus("Booked");
            aptReq1 = datalayer.saveAppointmentRequest(aptReq1);
            VARAppointmentRequests requests = datalayer.fetchAppointmentRequestsByPatient(patientId, null);
            assertEquals(1, requests.size());

            //submit second request, should go on top
            aptReq2 = createBasicAppointmentRequest(patientId, "testUser", "testLastName", "123232345", new DOBDate(new Date()));
            aptReq2.setCreatedDate(new Date());
            aptReq2.setStatus("Submitted");
            aptReq2 = datalayer.saveAppointmentRequest(aptReq2);
            requests = datalayer.fetchAppointmentRequestsByPatient(patientId, null);
            assertEquals(2, requests.size());
            assertEquals(aptReq2.getAppointmentRequestId(), requests.get(0).getAppointmentRequestId());

            //provider sends message to booked appointment. Should go top of list.
            VARAppointmentRequestMessage appRequestMessage = createAppointmentRequestMessage(aptReq1.getAppointmentRequestId());
            appRequestMessage.setPatientId(patientId);
            datalayer.saveAppointmentRequestMessage(appRequestMessage.getPatientId(), appRequestMessage, true);
            requests = datalayer.fetchAppointmentRequestsByPatient(patientId, null);
            assertEquals(2, requests.size());
            assertEquals(aptReq1.getAppointmentRequestId(), requests.get(0).getAppointmentRequestId());

            //Cancel submitted request. It should still be second
            aptReq2.setStatus("Cancelled");
            aptReq2.setLastUpdatedDate(new Date());
            aptReq2 = datalayer.saveAppointmentRequest(aptReq2);
            requests = datalayer.fetchAppointmentRequestsByPatient(patientId, null);
            assertEquals(2, requests.size());
            assertEquals(aptReq2.getAppointmentRequestId(), requests.get(1).getAppointmentRequestId());

            //veteran reads message, should now go second.
            datalayer.updateMessageFlag(aptReq1.getPatientId(), aptReq1.getAppointmentRequestId(), false);
            requests = datalayer.fetchAppointmentRequestsByPatient(patientId, null);
            assertEquals(aptReq1.getAppointmentRequestId(), requests.get(1).getAppointmentRequestId());

        }
        finally{
            if(aptReq1 != null){
                datalayer.deleteAppointmentRequest(aptReq1);
            }

            if(aptReq2 != null){
                datalayer.deleteAppointmentRequest(aptReq2);
            }

        }

    }

    private void checkAfterReadingMessageRequestGoesBackToBottom() {
        appointmentRequest4 = datalayer.updateMessageFlag(appointmentRequest4.getPatientId(), appointmentRequest4.getAppointmentRequestId(), true);
        VARAppointmentRequests aptRequests = datalayer.fetchAppointmentRequests(null);
        assertEquals(appointmentRequest4.getAppointmentRequestId(), aptRequests.get(aptRequests.size()-1).getUniqueId());
    }

    private void checkUnreadMessageAndRequestReadBeforeSendsRequestToTopOfList() {
    	VARAppointmentRequestMessage appRequestMessage = createAppointmentRequestMessage(appointmentRequest4.getAppointmentRequestId());
        datalayer.saveAppointmentRequestMessage(appRequestMessage.getPatientId(), appRequestMessage, false);

        appointmentRequest4 = datalayer.fetchAppointmentRequest(testPatientId, appointmentRequest4.getUniqueId());
        appointmentRequest4 = datalayer.markAppointmentRequestIsSeenByProvider(appointmentRequest4.getPatientId(), appointmentRequest4.getUniqueId());
        VARAppointmentRequests aptRequests = datalayer.fetchAppointmentRequests(null);
        assertEquals(appointmentRequest4.getAppointmentRequestId(), aptRequests.get(0).getUniqueId());
    }

    private void checkUnreadMessageAndRequestNeverReadBeforeSendsRequestToBottomOfList() {
    	VARAppointmentRequestMessage appRequestMessage = createAppointmentRequestMessage(appointmentRequest4.getAppointmentRequestId());
        datalayer.saveAppointmentRequestMessage(appRequestMessage.getPatientId(), appRequestMessage, false);
        appointmentRequest4 = datalayer.fetchAppointmentRequest(testPatientId, appointmentRequest4.getUniqueId());
        VARAppointmentRequests aptRequests = datalayer.fetchAppointmentRequests(null);
        assertEquals(appointmentRequest4.getAppointmentRequestId(), aptRequests.get(aptRequests.size()-1).getUniqueId());
    }

    private VARAppointmentRequestMessage createAppointmentRequestMessage(String appointmentRequestId) {
    	VARAppointmentRequestMessage mockAptRequestMessage = new VARAppointmentRequestMessage();
        mockAptRequestMessage.setSenderId("1");
        mockAptRequestMessage.setUniqueId("1");
        mockAptRequestMessage.setPatientId(testPatientId);
        mockAptRequestMessage.setMessageDateTime(new Date());
        mockAptRequestMessage.setAppointmentRequestId(appointmentRequestId);
        mockAptRequestMessage.setMessageText("Hi");
        return mockAptRequestMessage;
    }

    private void checkNewAppointmentPresentAndLastOnList() {
    	VARAppointmentRequests aptRequests = datalayer.fetchAppointmentRequests(null);
        assertEquals(appointmentRequest4.getAppointmentRequestId(), aptRequests.get(aptRequests.size()-1).getUniqueId());
    }


    private void saveAndCheckBasicAppointmentRequest(String patientId) {
		if (NullChecker.isNullish(patientId)) {
			patientId = UUID.randomUUID().toString();
		}

		appointmentRequest = createBasicAppointmentRequest(patientId, "testUser", "testLastName", "123232345", new DOBDate(new Date()));
		appointmentRequest = datalayer.saveAppointmentRequest(appointmentRequest);

		verifyAppointmentRequest(appointmentRequest);
	}

	private void verifyAppointmentRequest(VARAppointmentRequest appointmentRequest) {
		assertNotNull(appointmentRequest);
		assertNotNull(appointmentRequest.getUniqueId());
		assertEquals(appointmentType, appointmentRequest.getAppointmentType());
		assertEquals(visitType, appointmentRequest.getVisitType());
		assertEquals(facilityName, appointmentRequest.getFacility().getName());
		assertEquals(email, appointmentRequest.getEmail());
		assertEquals(phoneNumber, appointmentRequest.getPhoneNumber());
		assertEquals(status, appointmentRequest.getStatus());
		assertEquals(DateHelper.formatDate(createdDate), DateHelper.formatDate(appointmentRequest.getCreatedDate()));
		assertEquals(optionDate1, appointmentRequest.getOptionDate1());
		assertEquals(optionTime1, appointmentRequest.getOptionTime1());
		assertEquals(optionDate2, appointmentRequest.getOptionDate2());
		assertEquals(optionTime2, appointmentRequest.getOptionTime2());
		assertEquals(optionDate3, appointmentRequest.getOptionDate3());
		assertEquals(optionTime3, appointmentRequest.getOptionTime3());
		assertTrue(appointmentRequest.getBestTimetoCall().size() > 0);
		assertEquals(bestTimetoCall.toString(), appointmentRequest.getBestTimetoCall().toString());
		assertEquals(purposeOfVisit, appointmentRequest.getPurposeOfVisit());
		assertEquals(otherPurposeOfVisit, appointmentRequest.getOtherPurposeOfVisit());
		assertEquals(providerId, appointmentRequest.getProviderId());
		assertEquals(providerName, appointmentRequest.getProviderName());
		assertEquals(providerPersonClass, appointmentRequest.getProviderPersonClass());
		assertEquals(providerOption, appointmentRequest.getProviderOption());
		assertTrue(NullChecker.isNotNullish(appointmentRequest.getPatient().getFirstName()));
		assertTrue(NullChecker.isNotNullish(appointmentRequest.getPatient().getLastName()));
		assertEquals(secondRequest, appointmentRequest.isSecondRequest());
	}

	@Test
	public void verifyAppointmentFailsWhenFewFieldsAreMissingData(){
		VARAppointmentRequest saved=null;
		String patientId = UUID.randomUUID().toString();
		appointmentRequest = createBasicAppointmentRequest(patientId, "testUser", "testLastName", "123232345", new DOBDate(new Date()));
		appointmentRequest.setProviderName(AppointmentRequestProviderOption.WILLING_TO_SEE_OTHER_PROVIDER.getName());
		appointmentRequest.setRequestedPhoneCall(false);

		try{
			appointmentRequest.setProviderOption(null);
			saved = appointmentRequestDataService.saveAppointmentRequest(null , appointmentRequest, "", true, null);
		}
		catch (InvalidAppointmentRequestException e) {
		}
		assertNull(saved);

		try{
			appointmentRequest.setProviderOption("Call before booking an appointment");
			appointmentRequest.setVisitType(null);
			saved = appointmentRequestDataService.saveAppointmentRequest(null , appointmentRequest, "", true, null);
		}
		catch (InvalidAppointmentRequestException e) {
		}
		assertNull(saved);

		try{
			appointmentRequest.setProviderOption("Call before booking an appointment");
			appointmentRequest.setVisitType(visitType);
			appointmentRequest.setPurposeOfVisit(null);
			saved = appointmentRequestDataService.saveAppointmentRequest(null , appointmentRequest, "", true, null);
		}
		catch (InvalidAppointmentRequestException e) {
		}
		assertNull(saved);

		appointmentRequest.setRequestedPhoneCall(true);
		appointmentRequest.setPurposeOfVisit(AppointmentRequestPurposeOfVisit.NEW_PROBLEM.getName());
		saved =  appointmentRequestDataService.saveAppointmentRequest(null , appointmentRequest, "", true, null);
		assertNotNull(saved);
	}
}